<?php
/**
 * Reformer for Elementor
 * Customizable contact form for Elementor editor
 * Exclusively on https://1.envato.market/reformer-elementor
 *
 * @encoding        UTF-8
 * @version         1.0.2
 * @copyright       (C) 2018 - 2021 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Dmitry Merkulov (dmitry@merkulov.design), Cherviakov Vlad (vladchervjakov@gmail.com)
 * @support         help@merkulov.design
 **/

namespace Merkulove\ReformerElementor;

use Elementor\Plugin;
use Merkulove\ReformerElementor\Unity\Settings;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * SINGLETON: Caster class contain main plugin logic.
 *
 * @since 1.0.0
 *
 **/
final class Caster {

	/**
	 * The one true Caster.
	 *
     * @since 1.0.0
     * @access private
	 * @var Caster
	 **/
	private static $instance;


    const POST_TYPE = 'mdp_reformer_sub';

    /**
     * Setup the plugin.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function setup() {

        /** Define hooks that runs on both the front-end as well as the dashboard. */
        $this->both_hooks();

        /** Define public hooks. */
        $this->public_hooks();

        /** Define admin hooks. */
        $this->admin_hooks();

        /** Create directory for uploads. */
        wp_mkdir_p( trailingslashit( wp_upload_dir()['basedir'] ) . 'reformer' );

    }

    /**
     * Define hooks that runs on both the front-end as well as the dashboard.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function both_hooks() {

        /** Hooks which should work everywhere in backend and frontend. */

        /** Add Ajax Callback */
        add_action( 'wp_ajax_mdp_reformer_send', [$this, 'mdp_form_submit'] );
        add_action( 'wp_ajax_nopriv_mdp_reformer_send', [$this, 'mdp_form_submit'] );

    }

    /**
     * Register all of the hooks related to the public-facing functionality.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function public_hooks() {

        /** Work only on frontend area. */
        if ( is_admin() ) { return; }

        /** Public hooks and filters. */

    }

    /**
     * Register all of the hooks related to the admin area functionality.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function admin_hooks() {

        /** Work only in admin area. */
        if ( ! is_admin() ) { return; }

        /** Admin hooks and filters here. */

        /** Create CPT */
        add_action( 'init', [ $this, 'create_cpt' ] );
        add_action( 'init', [ $this, 'create_address_book_cpt' ] );

        /** Add custom metabox */
        add_action( 'add_meta_boxes', [ $this, 'add_meta_box' ] );

        /** Setup custom metabox */
        /** Fire meta box setup on the post editor screen. */
        add_action( 'load-post.php', [ $this, 'setup_address_book_meta_box' ] );
        add_action( 'load-post-new.php', [ $this, 'setup_address_book_meta_box' ] );

        /** Change title column name in address book post type */
        add_filter( 'manage_mdp_address_book_cpt_posts_columns', [ $this, 'change_address_book_cpt_title_column_name' ] );

        /** Change title placeholder in address book post type */
        add_filter( 'enter_title_here', [ $this, 'change_address_book_title_placeholder' ], 10, 2 );

        /** Styles and scripts for settings page */
        AdminStyles::get_instance();
        AdminScripts::get_instance();

        /** Remove admin menu added from Unity. */
        add_action( 'admin_menu', [ $this, 'remove_admin_menu' ], 1000 );
        add_action( 'admin_menu', [ $this, 'add_admin_menu' ], 1001 );

        /** Notice if Elementor not installed */
        if ( !did_action( 'elementor/loaded' ) ) {
            add_action('admin_footer', [$this, 'elementor_activation_notice']);
        }

    }

	/**
	 * This method used in register_activation_hook
	 * Everything written here will be done during plugin activation.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function activation_hook() {

		/** Activation hook */

	}


    /**
     * Show admin warning, if Elementor not installed or not activated.
     *
     * @since 1.0.0
     * @access public
     **/
    public function elementor_activation_notice() {
        $screen = get_current_screen();

        if ( $screen->id === 'edit-mdp_reformer_sub' ||
            $screen->id === 'mdp_reformer_sub' ||
            $screen->id === 'mdp_address_book_cpt' ||
            $screen->id === 'edit-mdp_address_book_cpt' ) {

            echo sprintf(
        '<div class="settings-error notice notice-warning"><p><strong>%s</strong></p></div>',
                esc_html__( "This plugin requires Elementor. Please install Elementor.", "reformer-elementor" )
            );

        }
    }

    /**
     * Add admin menu for plugin settings.
     *
     * @since 1.0.0
     * @access public
     **/
    public function add_admin_menu() {

        add_submenu_page(
            'edit.php?post_type=' . self::POST_TYPE,
            esc_html__( 'ReFormer Settings', 'reformer-elementor' ),
            esc_html__( 'Settings', 'reformer-elementor' ),
            'manage_options',
            'mdp_reformer_elementor_settings',
            [Settings::get_instance(), 'options_page']
        );

    }

    /**
     * Remove admin menu added from Unity.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    public function remove_admin_menu() {

        /** Check if Elementor installed and activated. */
        $parent = 'options-general.php';
        if ( did_action( 'elementor/loaded' ) ) {
            $parent = 'elementor';
        }

        /** Remove submenu item by slug. */
        remove_submenu_page(  $parent, 'mdp_reformer_elementor_settings' );

    }


    /**
     * This method for sanitization form fields.
     *
     * @return string
     * @since 1.0.0
     * @access private
     *
     */
    private function sanitize_field ( $value, $type ) {
        switch ( $type ) {
            case 'text':
            case 'hidden':
            case 'checkbox':
            case 'radio':
            case 'select':
                $value = sanitize_text_field( $value );
                break;
            case 'url':
                $value = esc_url_raw( $value );
                break;
            case 'textarea':
                $value = sanitize_textarea_field( $value );
                break;
            case 'email':
                $value = sanitize_email( $value );
                break;
            default:
                $value = esc_html( $value );
        }

        return $value;
    }


    /**
     * Form submit method.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function mdp_form_submit() {

        if ( empty( $_POST['mdp_reformer_nonce'] ) ||
            ! wp_verify_nonce( $_POST['mdp_reformer_nonce'], 'reformer-elementor-nonce' ) ) { return; }

        $form_data = [];

        wp_parse_str( $_POST['mdp_reformer_send'], $form_data );

        $plugin_elementor = Plugin::instance();

        $form_id = null;
        $post_id = null;

        foreach ( $form_data as $form_field ) {
            if ( $form_field[1] === 'mdp_post_id' ) {
                $post_id = $form_field[2];
            }

            if ( $form_field[1] === 'mdp_form_id' ) {
                $form_id = $form_field[2];
            }
        }

        $post_id_data = $post_id;

        $document = $plugin_elementor->documents->get( $post_id );

        $form = null;

        if ( !empty( $document ) ) {
            $form = $this->find_form( $document->get_elements_data(), $form_id );
        }

        $form_settings = $form['settings'];

        $success_message = empty( $form_settings['success_message'] ) ?
            'Form sent!' : esc_html( $form_settings['success_message'] );

        $error_message = empty( $form_settings['error_message'] ) ?
            'Error!' : esc_html( $form_settings['error_message'] );

        if ( empty( $form ) ) {
            wp_send_json_error( $error_message );
        }

        $send_to = [ 'save', 'email' ];
        if ( isset( $form_settings['send_to'] ) ) {
            $send_to = $form_settings['send_to'];
        }

        // save files and get file urls
        $file_paths = [];

        if ( !empty( $send_to ) ) {
            foreach ( $form_data as $form_field ) {
                if ( $form_field[0] === 'file' ) {
                    $file_paths[ $this->sanitize_field( $form_field[1], $form_field[0] ) ] = $this->upload_file_field(
                        $form_id,
                        $this->sanitize_field( $form_field[3], $form_field[0] ),
                        $this->sanitize_field( $form_field[4], $form_field[0] ),
                        $this->sanitize_field( $form_field[1], $form_field[0] )
                    );
                }
            }
        }

        // save as cpt
        if ( in_array( 'save', $send_to ) ) {

            $record_suffix = '';

            foreach ( $form_data as $value ) {
                if ( $value[3] === $form_settings['record_suffix'] ) {
                    $record_suffix = ' > ' . $value[2];
                }
            }

            $form_suffix = !empty( $form_settings['form_name'] ) ? $form_settings['form_name'] : 'form_' . $form_id;

            $args = [];
            $args['post_type'] = 'mdp_reformer_sub';
            $args['post_title'] = $form_suffix . $record_suffix;
            $post_id = wp_insert_post( $args );
            foreach ( $form_data as $value ) {

                if ( $value[1] === 'mdp_post_id' ) { continue; }

                if ( $value[1] === 'mdp_form_id' ) { $value[1] = 'Form id'; }

                if ( empty( $value[2] ) ) { continue; }

                if ( $value[0] === 'file' && empty( $file_paths[ $this->sanitize_field( $value[1], $value[0] ) ]
                    [ $this->sanitize_field( $value[1], $value[0] ) ] ) ) { continue; }

                if ( $value[2] === 'mdp_reformer_file_send' ) {
                    add_post_meta( $post_id, $this->sanitize_field( $value[1], $value[0] ),
                        $file_paths[ $this->sanitize_field( $value[1], $value[0] ) ]
                            [ $this->sanitize_field( $value[1], $value[0] ) ] );

                } else {
                    add_post_meta( $post_id, $this->sanitize_field( $value[1], $value[0] ),
                        $this->sanitize_field( $value[2], $value[0] ) );
                }

                // add page url to submission
                if ( $form_settings['send_referrer'] === 'yes' ) {
                    add_post_meta( $post_id, 'Page url', esc_url( get_permalink( $post_id_data ) ) );
                }

            }
        }

        // send email
        if ( in_array( 'email', $send_to ) ) {
            $url_parts = parse_url( home_url() );
            $domain = $url_parts['host'];

            $email_to  = !empty( $form_settings['email_to'] ) && $form_settings['use_address_book_email'] !== 'yes' ?
                esc_html( $form_settings['email_to'] ) : get_option( 'admin_email' );

            if ( $form_settings['use_address_book_email'] === 'yes' ) {
                $email_to = get_post_meta( $form_settings['address_book_email'] )['mdp_field_contact_email'][0];
            }

            $email_subject = !empty( $form_settings['email_subject'] ) ?
                esc_html( $form_settings['email_subject'] ) : 'Message from '.get_bloginfo( 'name' );
            $email_content = str_replace(
             '[form-fields]',
                    $this->get_form_data_string( $form_data, $file_paths, $form_settings['send_referrer'], $post_id_data ),
                    esc_html__( $form_settings['email_message'] )
            );
            $email_from_name = !empty( $form_settings['email_from_name'] ) ?
                esc_html( $form_settings['email_from_name'] ) : get_bloginfo( 'name' );
            $email_reply_to = !empty( $form_settings['email_from_email'] ) ?
                esc_html( $form_settings['email_from_email'] ) : 'noreply@'.$domain;

            $headers = 'From: ' . $email_from_name . ' ' . $email_reply_to . "\r\n" .
                        'Reply-To:' . $email_reply_to . "\r\n";

            if ( !filter_var( $email_to, FILTER_VALIDATE_EMAIL ) ) {
                if ( current_user_can( 'manage_options' ) ) {
                    wp_send_json_error( 'Incorrect receiver email!' );
                } else {
                    wp_send_json_error( $error_message );
                }
            }


            wp_mail( $email_to, $email_subject, $email_content, $headers );
        }

        if ( in_array( 'discord', $send_to ) ) {

            $discord_webhook = $form_settings['discord_webhook'];

            if ( $form_settings['use_address_book_discord'] === 'yes' ) {
                $discord_webhook = get_post_meta( $form_settings['address_book_discord'] )['mdp_field_discord_webhook'][0];
            }

            if ( strpos( $discord_webhook, 'https://discord.com/api/webhooks/' ) === false &&
                strpos( $discord_webhook, 'https://discordapp.com/api/webhooks/' ) === false ) {
                if ( current_user_can( 'manage_options' ) ) {
                    wp_send_json_error( 'Incorrect Discord webhook!' );
                } else {
                    wp_send_json_error( $error_message );
                }
            }

            if ( empty( $form_settings['discord_webhook'] ) && $form_settings['use_address_book_discord'] !== 'yes' ) {
                if ( current_user_can( 'manage_options' ) ) {
                    wp_send_json_error( 'Discord webhook is empty!' );
                } else {
                    wp_send_json_error( $error_message );
                }
            }


            $endpoint = esc_url( $discord_webhook );


            $message_data = [
                'title' => !empty( $form_settings['discord_title'] ) ?
                    esc_html( $form_settings['discord_title'] ) :
                    esc_html__( 'A new submission', 'reformer-elementor' ),
                'description' => !empty( $form_settings['discord_description'] ) ?
                    esc_html( $form_settings['discord_description'] ) :
                    esc_html__( 'A new submission received', 'reformer-elementor' ),
                'author' => [
                    'name' => !empty( $form_settings['discord_username'] ) ?
                        esc_html( $form_settings['discord_username'] ) :
                        esc_html__( 'ReFormer', 'reformer-elementor' ),
                    'url' => site_url(),
                    'icon_url' => !empty( $form_settings['discord_avatar_url'] ) &&
                     filter_var( $form_settings['discord_avatar_url']['url'], FILTER_VALIDATE_URL ) ?
                        esc_url( $form_settings['discord_avatar_url']['url'] ) : null
                ],
                'url' => $form_settings['send_referrer'] === 'yes' ?
                        esc_url( get_permalink( $post_id_data ) ) :
                        esc_url( site_url() ),
                'color' => !empty( $form_settings['discord_color'] ) ?
                    hexdec( ltrim( esc_html__( $form_settings['discord_color'], '#' ) ) ) :
                    hexdec( '1cb4f2' )
            ];
            
            if ( $form_settings['discord_timestamp'] === 'yes' ) {
                $message_data['timestamp'] = gmdate( \DateTime::ISO8601 );
            }

            $form_send_fields = [];


            $send_fields = 'yes';
            if ( isset( $form_settings['discord_form_fields'] ) ) {
                $send_fields = $form_settings['discord_form_fields'];
            }

            if ( $send_fields === 'yes' ) {

                foreach ( $form_data as $value ) {

                    if ( $value[1] === 'mdp_post_id' || $value[1] === 'mdp_form_id' ) { continue; }

                    // check if value empty
                    if ( empty( $value[2] ) ) { continue; }

                    if ( $value[0] === 'file' && empty( $file_paths[ $this->sanitize_field( $value[1], $value[0] ) ]
                        [ $this->sanitize_field( $value[1], $value[0] ) ] ) ) { continue; }

                    $form_send_fields[] = [
                        'name' => esc_html( $value[1] ),
                        'value' => $value[0] === 'file' ?
                            $file_paths[ $this->sanitize_field( $value[1], $value[0] ) ]
                            [ $this->sanitize_field( $value[1], $value[0] ) ] :
                            $this->sanitize_field( $value[2], $value[0] ),
                        'inline' => false
                    ];

                }

                $message_data['fields'] = array_values( $form_send_fields );
            }


            $discord_webhook_data = [
                'embeds' => array_values( [$message_data] )
            ];

            $response = wp_remote_post( $endpoint, [
                'body' => wp_json_encode( $discord_webhook_data ),
                'headers' => [ 'Content-Type' => 'application/json; charset=utf-8' ],
            ] );

            if ( ( int ) wp_remote_retrieve_response_code( $response ) !== 204 ) {
                if ( current_user_can( 'manage_options' ) ) {
                    wp_send_json_error( 'Discord bad request!' );
                } else {
                    wp_send_json_error( $error_message );
                }
            }

        }

        if ( in_array( 'slack', $send_to ) ) {

            $slack_webhook = $form_settings['slack_webhook'];

            if ( $form_settings['use_address_book_slack'] === 'yes' ) {
                $slack_webhook = esc_url( get_post_meta( $form_settings['address_book_slack'] )
                                ['mdp_field_slack_webhook'][0] );
            }

            if ( strpos( $slack_webhook, 'https://hooks.slack.com/services/' ) === false ) {
                if ( current_user_can( 'manage_options' ) ) {
                    wp_send_json_error( 'Incorrect Slack webhook!' );
                } else {
                    wp_send_json_error( $error_message );
                }
            }

            if ( empty( $slack_webhook ) ) {
                if ( current_user_can( 'manage_options' ) ) {
                    wp_send_json_error( 'Slack webhook is empty!' );
                } else {
                    wp_send_json_error( $error_message );
                }
            }

            $webhook_slack_data = [
              'username' => !empty( $form_settings['slack_username'] ) ?
                  esc_html__( $form_settings['slack_username'] ) : '',
            ];

            $slack_attachment = [
                'text' => !empty( $form_settings['slack_description'] ) ?
                    esc_html( $form_settings['slack_description'] ) :
                    esc_html__( 'A new form submission has been received', 'reformer-elementor' ),
                'title' => !empty( $form_settings['slack_title'] ) ?
                    esc_html( $form_settings['slack_title'] ) :
                    esc_html__( 'A new form submission', 'reformer-elementor' ),
                'color' => !empty( $form_settings['slack_color'] ) ?
                    esc_html( $form_settings['slack_color'] ) : '#1CB4F2',
                'title_link' => $form_settings['send_referrer'] ?
                    esc_url( get_permalink( $post_id_data ) ) :
                    esc_url( site_url() ),
            ];

            $form_send_fields = [];

            $send_fields = 'yes';
            if( isset( $form_settings['slack_form_fields'] ) ) {
                $send_fields = $form_settings['slack_form_fields'];
            }

            if ( $send_fields ) {

                foreach ( $form_data as $value ) {

                    // check if value empty
                    if ( empty( $value[2] ) ) { continue; }

                    if ( $value[0] === 'file' && empty( $file_paths[ $this->sanitize_field( $value[1], $value[0] ) ]
                        [ $this->sanitize_field( $value[1], $value[0] ) ] ) ) { continue; }

                    if ( $value[1] === 'mdp_post_id' || $value[1] === 'mdp_form_id' ) { continue; }

                    $form_send_fields[] = [
                        'title' => esc_html( $value[1] ),
                        'value' => $value[0] === 'file' ?
                            $file_paths[ $this->sanitize_field( $value[1], $value[0] ) ]
                                    [ $this->sanitize_field( $value[1], $value[0] ) ] :
                            $this->sanitize_field( $value[2], $value[0] ),
                        'short' => false
                    ];

                }

                $slack_attachment['fields'] = $form_send_fields;
            }

            if ( !empty( $form_settings['slack_timestamp'] ) ) {
                $slack_attachment['ts'] = time();
            }

            $webhook_slack_data['attachments'] = [ $slack_attachment ];

            $response = wp_remote_post( $slack_webhook, [
                'headers' => [
                    'Content-Type' => 'application/json',
                ],
                'body' => wp_json_encode( $webhook_slack_data ),
            ] );

            if ( ( int ) wp_remote_retrieve_response_code( $response ) !== 200 ) {
                if ( current_user_can( 'manage_options' ) ) {
                    wp_send_json_error( 'Slack bad request!' );
                } else {
                    wp_send_json_error( $error_message );
                }
            }

        }

        if ( in_array( 'telegram', $send_to ) ) {

            $options = Settings::get_instance()->options;

            $token = esc_html( $options['telegram_bot_token'] );
            $method = 'sendMessage';

            if ( empty( $token ) ) {
                if ( current_user_can( 'manage_options' ) ) {
                    wp_send_json_error( 'Telegram token is empty!' );
                } else {
                    wp_send_json_error( $error_message );
                }
            }

            if ( empty( get_option( 'mdp_telegram_chats' ) ) ) {
                if ( current_user_can( 'manage_options' ) ) {
                    wp_send_json_error( 'Please choose telegram chat!' );
                } else {
                    wp_send_json_error( $error_message );
                }
            }

            $chat_id = !isset( $form_settings['telegram_chat'] ) && !empty( get_option( 'mdp_telegram_chats' ) ) ?
                       array_keys( get_option( 'mdp_telegram_chats' ) )[0] :
                       $form_settings['telegram_chat'];


            $bot_message_parameters = [
                'chat_id' => esc_html( $chat_id ),
                'text' => str_replace(
                '[form-fields]',
                       $this->get_form_data_string( $form_data, $file_paths, $form_settings['send_referrer'], $post_id_data ),
                       esc_html( $form_settings['telegram_message'] ) )
            ];

            $url = 'https://api.telegram.org/bot'.$token.'/'.$method;


            $response = wp_remote_post( $url, [
                'headers' => [
                    'Content-Type' => 'application/json',
                ],
                'body' => wp_json_encode( $bot_message_parameters )
            ] );

        }


        wp_send_json_success( $success_message );


    }

    /**
     * Callback for creating form fields metabox.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function form_fields_metabox_callback() {
        add_meta_box(
            'mdp_reformer_metabox',
                esc_html__( 'Form Fields', 'reformer-elementor' ),
                [ $this, 'form_fields_metabox' ],
            '',
            'normal',
            'high'
        );
    }

    /**
     * Form fields data metabox.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function form_fields_metabox( $post ) {
        $post_meta = get_post_meta( $post->ID );
        ?>
        <table class="form-table">
            <tbody>
        <?php
        foreach ( $post_meta as $key => $value ):
            if ( $key === '_edit_lock' || $key === '_edit_last' ) { continue; }
            ?>

        <tr>
            <th scope="row">
                <label><?php esc_html_e( $key.': ' ); ?></label>
            </th>
            <td>
                <div>
                    <div class="mdp-reformer-elementor-field-value">
                        <?php if( filter_var( $value[0], FILTER_VALIDATE_URL ) ): ?>
                        <a href="<?php echo esc_url( $value[0] ) ?>" target="_blank"><?php esc_html_e( $value[0] ); ?></a>
                        <?php else: ?>
                        <span><?php esc_html_e( $value[0] ); ?></span>
                        <?php endif; ?>
                    </div>
                </div>
            </td>
        </tr>
        <?php endforeach; ?>
            </tbody>
        </table>
        <?php
    }

    /**
     * Upload files method.
     *
     * @return array|false
     * @since 1.0.0
     * @access public
     *
     */
    private function upload_file_field( $form_id, $allowed_types, $limit_size, $field_name ) {
        if ( empty( $_FILES ) ) { return false; } // Exit no file

        $file_urls = [];

        $file_mime = mime_content_type( $_FILES[$field_name]['tmp_name'] );
        $file_size = filesize( $_FILES[$field_name]['tmp_name'] );
        $extensions_array = explode( '.', $_FILES[$field_name]['name'] );
        $file_extension = array_pop( $extensions_array );

        if ( !$file_mime || !$file_size ) { return false; }

        // Limits
        $limit_mime = explode( ',', $allowed_types );
        $limit_mime = array_map( 'trim', $limit_mime );

        // Remove file id mime type is incorrect
        if ( ! in_array( strtolower( $file_extension ), $limit_mime ) && $limit_mime[ 0 ] !== '*' ) {

            wp_delete_file( $_FILES[$field_name][ 'tmp_name' ] );
            wp_send_json_error( 'The type of the attached file does not match allowed types.' );

        }

        // Remove file id file size larger than file size limit
        if ( intval( $limit_size ) * pow( 1024, 2 ) < $file_size ) {

            wp_delete_file( $_FILES[$field_name][ 'tmp_name' ] );
            wp_send_json_error( 'File size is too large!' );

        }

        $file_data = $this->create_unique_file_name( $form_id, $file_extension );

        file_put_contents( $file_data['path'], file_get_contents( $_FILES[$field_name]['tmp_name'] ), FILE_APPEND );

        wp_delete_file( $_FILES[$field_name]['tmp_name'] );

        $file_urls[$field_name] = $file_data['url'];

        return $file_urls;

    }

    /**
     * Method for creating file name.
     *
     * @return array
     * @since 1.0.0
     * @access public
     *
     */
    private function create_unique_file_name( $form_id, $ext ) {
        $upload_dir = wp_get_upload_dir();
        $upload_basedir = $upload_dir['basedir'] . '/reformer/';

        $file_data = [];

        $counter = 0;

        $file_name = 'reformer-' . $form_id . '-' . gmdate( 'Y-m-d\TH:i:s\Z' ) . '-' . $counter . '.' .$ext;

        $f_path = $upload_basedir . $file_name;

        if ( file_exists( $f_path ) ) {
            do {
                $counter++;
                $file_name = 'reformer-' . $form_id . '-' . gmdate( 'Y-m-d\TH:i:s\Z' ) . '-' . $counter . '.' .$ext;
                $f_path = $upload_basedir . $file_name;
            } while ( file_exists( $f_path ) );
        }

        $f_path = wp_normalize_path( $f_path );

        $file_data['path'] = str_replace( ['/', '\\'], DIRECTORY_SEPARATOR, $f_path );
        $file_data['url'] = $upload_dir['baseurl'] . '/reformer/' . $file_name;

        return $file_data;

    }

    /**
     * Make form fields array to string.
     *
     * @param $form_data
     * @param $file_paths
     * @param $enable_page_url
     * @param $post_id
     * @return string
     * @since 1.0.0
     * @access public
     */
    private function get_form_data_string( $form_data, $file_paths, $enable_page_url, $post_id ) {

        $form_fields = '';

        foreach ( $form_data as $value ) {

            if ( $value[1] === 'mdp_post_id' || $value[1] === 'mdp_form_id' ) { continue; }

            if ( empty( $value[2] ) ) { continue; }

            if ( $value[0] === 'file' && empty( $file_paths[ $this->sanitize_field( $value[1], $value[0] ) ]
                [ $this->sanitize_field( $value[1], $value[0] ) ] ) ) { continue; }

            if ( $value[2] === 'mdp_reformer_file_send' ) {
                $form_fields .= esc_html( $value[1] ) . " - " .
                    $file_paths[ $this->sanitize_field( $value[1], $value[0] ) ]
                    [ $this->sanitize_field( $value[1], $value[0] ) ] . "\n";
            } else {
                $form_fields .= esc_html( $value[1] ) . " - " . $this->sanitize_field( $value[2], $value[0] ) . "\n";
            }

        }

        if ( $enable_page_url === 'yes' ) {
            $form_fields .= esc_html__( 'Page url' ) . " - " . esc_url( get_permalink( $post_id ) );
        }

        return $form_fields;

    }


    /**
     * Find form widget method.
     *
     * @param $elements
     * @param $form_id
     * @return bool
     * @since 1.0.0
     * @access public
     *
     */
    private function find_form( $elements, $form_id ) {
        foreach ( $elements as $element ) {
            if ( $form_id === $element['id'] ) {
                return $element;
            }

            if ( ! empty( $element['elements'] ) ) {
                $element = self::find_form( $element['elements'], $form_id );

                if ( $element ) {
                    return $element;
                }
            }
        }

        return false;
    }

    /**
     * Return path to admin menu icon or base64 encoded image.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string
     **/
    private function get_admin_menu_icon() {

        return 'data:image/svg+xml;base64,' . base64_encode(
                file_get_contents( Unity\Plugin::get_path() . 'images/logo-menu.svg' )
            );

    }


    /**
     * Create Address Book Custom Post Type method.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function create_address_book_cpt() {
        register_post_type( 'mdp_address_book_cpt', [
            'public'              => false,
            'labels'              => [
                'name'                  => esc_html__( 'Address Book', 'reformer-elementor' ),
                'singular_name'         => esc_html__( 'Address Book', 'reformer-elementor' ),
                'add_new'               => esc_html__( 'Add New', 'reformer-elementor' ),
                'add_new_item'          => esc_html__( '', 'reformer-elementor' ),
                'new_item'              => esc_html__( 'New Address Book Contact', 'reformer-elementor' ),
                'edit_item'             => esc_html__( 'Edit Address Book Contact', 'reformer-elementor' ),
                'view_item'             => esc_html__( 'View Address Book Contact', 'reformer-elementor' ),
                'view_items'            => esc_html__( 'View Address Book Contacts', 'reformer-elementor' ),
                'search_items'          => esc_html__( 'Search Address Book Contacts', 'reformer-elementor' ),
                'not_found'             => esc_html__( 'No Address Book Contacts found', 'reformer-elementor' ),
                'not_found_in_trash'    => esc_html__( 'No Address Book Contacts found in Trash', 'reformer-elementor' ),
                'all_items'             => esc_html__( 'Address Book', 'reformer-elementor' ),
                'archives'              => esc_html__( 'Address Book Contacts Archives', 'reformer-elementor' ),
                'attributes'            => esc_html__( 'Address Book Contacts Attributes', 'reformer-elementor' ),
                'insert_into_item'      => esc_html__( 'Insert to Address Book Contact', 'reformer-elementor' ),
                'uploaded_to_this_item' => esc_html__( 'Uploaded to this Address Book Contacts', 'reformer-elementor' ),
                'menu_name'             => esc_html__( 'Address Book', 'reformer-elementor' ),
            ],
            'exclude_from_search'   => true,
            'show_in_menu'          => 'edit.php?post_type='.self::POST_TYPE,
            'publicly_queryable'    => false,
            'menu_position'         => false,
            'show_in_rest'          => false,
            'rest_base'             => 'mdp_address_book_cpt',
            'supports'              => [ 'title' ],
            'capabilities'          => [ 'create_posts' => true, 'publish_posts' => true ],
            'map_meta_cap'          => true,
            'show_ui'               => true,
        ] );
    }

    /**
     * Create Custom Post Type method.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function create_cpt() {
        register_post_type( self::POST_TYPE, [
            'public'              => false,
            'labels'              => [
                'name'                  => esc_html__( 'ReFormer Submissions', 'reformer-elementor' ),
                'singular_name'         => esc_html__( 'ReFormer Submission', 'reformer-elementor' ),
                'add_new'               => esc_html__( 'Add New', 'reformer-elementor' ),
                'add_new_item'          => esc_html__( 'Add New', 'reformer-elementor' ),
                'new_item'              => esc_html__( 'New ReFormer Submission', 'reformer-elementor' ),
                'edit_item'             => esc_html__( 'Edit ReFormer Submission', 'reformer-elementor' ),
                'view_item'             => esc_html__( 'View ReFormer Submission', 'reformer-elementor' ),
                'view_items'            => esc_html__( 'View ReFormer Submission', 'reformer-elementor' ),
                'search_items'          => esc_html__( 'Search ReFormer Submissions', 'reformer-elementor' ),
                'not_found'             => esc_html__( 'No ReFormer Submissions found', 'reformer-elementor' ),
                'not_found_in_trash'    => esc_html__( 'No ReFormer Submissions found in Trash', 'reformer-elementor' ),
                'all_items'             => esc_html__( 'Submissions', 'reformer-elementor' ),
                'archives'              => esc_html__( 'ReFormer Submissions Archives', 'reformer-elementor' ),
                'attributes'            => esc_html__( 'ReFormer Submissions Attributes', 'reformer-elementor' ),
                'insert_into_item'      => esc_html__( 'Insert to ReFormer Submissions', 'reformer-elementor' ),
                'uploaded_to_this_item' => esc_html__( 'Uploaded to this ReFormer Submissions', 'reformer-elementor' ),
                'menu_name'             => esc_html__( 'ReFormer', 'reformer-elementor' ),
            ],
            'menu_icon'             => $this->get_admin_menu_icon(),
            'exclude_from_search'   => true,
            'publicly_queryable'    => false,
            'menu_position'         => false,
            'show_in_rest'          => false,
            'rest_base'             => 'mdp_reformer_sub',
            'supports'              => [ 'title' ],
            'register_meta_box_cb' => [ $this, 'form_fields_metabox_callback' ],
            'capabilities'          => [ 'create_posts' => false, 'publish_posts' => false ],
            'map_meta_cap'          => true,
            'show_ui'               => true,
        ] );
    }

    /**
     * Add custom metabox.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function add_meta_box() {
        add_meta_box(
            'mdp-address-book-options',
                esc_html__( 'Address Book Contacts', 'reformer-elementor' ),
               [ $this, 'options_address_book_meta_box' ],
        'mdp_address_book_cpt',
       'normal',
       'default'
        );
    }

    /**
     * Changes address book cpt title column name.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function change_address_book_cpt_title_column_name( $columns ) {
        $columns['title'] = 'Contact Name';

        return $columns;
    }

    /**
     * Changes address book cpt title placeholder.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function change_address_book_title_placeholder( $title, $post ) {
        if ( $post->post_type === 'mdp_address_book_cpt' ) {
            $title = 'Contact Name';
        }

        return $title;
    }

    /**
     * Saves metabox.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function save_address_book_meta_box( $post_id, $post ) {
        /** Work only with mdp_address_book_cpt post type. */
        if ( 'mdp_address_book_cpt' !== $post->post_type ) { return; }

        /** Verify the nonce before proceeding. */
        if (
            ! isset( $_POST['options_metabox_fields_nonce'] ) ||
            ! wp_verify_nonce( $_POST['options_metabox_fields_nonce'], Unity\Plugin::get_basename() )
        ) {
            return;
        }

        /** Get the post type object. */
        $post_type = get_post_type_object( $post->post_type );

        /** Check if the current user has permission to edit the post. */
        if ( ! current_user_can( $post_type->cap->edit_post, $post_id ) ) {
            return;
        }

        /** Save "Options" metabox with all fields. */
        AddressBookMetaBoxOptions::get_instance()->save_metabox( $post_id );

    }

    /**
     * Render metabox method.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function options_address_book_meta_box( $reformer_form ) {
        AddressBookMetaBoxOptions::get_instance()->render_meta_box( $reformer_form );
    }

    /**
     * Setup metabox method.
     *
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public function setup_address_book_meta_box() {
        /** Add meta boxes on the 'add_meta_boxes' hook. */
        add_action( 'add_meta_boxes', [ $this, 'add_meta_box' ] );

        /** Save meta box values on the 'save_post' hook. */
        add_action( 'save_post_mdp_address_book_cpt', [ $this, 'save_address_book_meta_box' ], 1, 2 );
    }

	/**
	 * Main Caster Instance.
	 * Insures that only one instance of Caster exists in memory at any one time.
	 *
	 * @static
     * @since 1.0.0
     * @access public
     *
	 * @return Caster
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}
